# Abstract base class for actors (used both by classical RL algorithms and EAs) based on PyTorch.

from abc import abstractmethod
from models.torch_models import BaseTorchModel
from utils.types import *
import numpy as np
import torch


class BaseActor(BaseTorchModel):
    """Base class for the actors based on PyTorch able to act in environments modeled by Markov Decision Processes."""
    
    @abstractmethod
    def act(self, observation: ObservationType, explore: bool = False) -> ActionType:
        """
        Given an observation, return an action. (Possibly select exploratory action.)
        """
        pass
    
    @abstractmethod
    def act_batch(self, observations: torch.Tensor) -> torch.Tensor:
        """
        Given a batch of observations, return a batch of actions.
        """
        pass
    
    @abstractmethod
    def set_exploration_parameters(self, **exploration_parameters) -> None:
        """
        Set the exploration parameters for the actor.
        """
        pass

    @abstractmethod
    def vectorize(self) -> np.ndarray:
        """
        Return the parameters of the actor as a flat array of floats.
        """
        pass
    
    @abstractmethod
    def set_parameters(self, parameters: np.ndarray) -> None:
        """
        Set the parameters of the actor from a flat array of floats.
        """
        pass
    
    def reset(self) -> None:
        """
        Reset the internal state of the actor (if any) for a new episode.
        """
        pass


class BaseContinuousDeterministicActor(BaseActor):
    """
    Base class for continuous deterministic actors.
    """
    _constraints: Constraints = Constraints()
    exploration_noise_std: float = 0.0

    @property
    def constraints(self) -> Constraints:
        """
        Return the constraints of the actor's actions.
        """
        return self._constraints

    def set_exploration_parameters(self, *, exploration_noise_std: float) -> None:
        self.exploration_noise_std = exploration_noise_std
