# Class for Gymnasium environments.

from .base_env import BaseEnvironment
from utils.types import *
import gymnasium


class GymnasiumEnvironment(BaseEnvironment):
    """
    Wrapper for Gymnasium environments.
    """
    def __init__(self, env: gymnasium.Env) -> None:
        """
        Initialize the Gymnasium environment wrapper.
        """
        self.env = env


    @property
    def timestep_limit(self) -> int | None:
        return self.env.spec.max_episode_steps if self.env.spec else None


    def reset(self) -> tuple[ObservationType, dict[str, Any]]:
        observation, info = self.env.reset()
        return observation, info


    def step(self, action: ActionType) -> tuple[ObservationType, float, bool, bool, dict[str, Any]]:
        observation, reward, terminated, truncated, info = self.env.step(action)
        return observation, float(reward), terminated, truncated, info


    def set_seed(self, seed: int | None) -> None:
        self.env.reset(seed=seed)
        if hasattr(self.env, "action_space") and hasattr(self.env.action_space, "seed"):
            self.env.action_space.seed(seed)
        if hasattr(self.env, "observation_space") and hasattr(self.env.observation_space, "seed"):
            self.env.observation_space.seed(seed)
