# Abstract base class for Evolutionary Algorithms (EAs) in Reinforcement Learning.

from abc import ABC, abstractmethod
from utils.types import *
from models.actors.base_actor import BaseActor
from replay_buffers.base_replay_buffer import BaseReplayBuffer
from envs.base_env import BaseEnvironment


class BaseEA(ABC):
    """
    Abstract base class for Evolutionary Algorithms (EAs) in Reinforcement Learning.
    """
    def __init__(
        self,
        environment: BaseEnvironment,
        template_actor: BaseActor,
        replay_buffer: BaseReplayBuffer | None = None,
        *,
        seed: int | None = None,
        **hyperparameters: Any
    ) -> None:
        """
        Initialize the EA algorithm with the environment, actor, possibly a replay buffer, and hyperparameters.

        The provided actor is copied and the copy is used as a template for the actors for evolution
        via the create_sibling method.
        
        The replay buffer is used to store transitions encountered during the evolution process
        for usage in the classical RL algorithm, if provided.
        """
        self._environment = environment
        self._template_actor = template_actor.copy()
        self._replay_buffer = replay_buffer
        
        self._seed = seed
        
        hyperparameters.update({"seed": seed})
        self._hyperparameters = hyperparameters

    @property
    def environment(self) -> BaseEnvironment:
        """
        Return the environment used by the EA.
        """
        return self._environment
    
    @property
    def replay_buffer(self) -> BaseReplayBuffer | None:
        """
        Return the associated replay buffer used by the EA to store its transitions.
        """
        return self._replay_buffer

    @property
    def hyperparameters(self) -> dict[str, Any]:
        """
        Return the hyperparameters of the EA.
        """
        return self._hyperparameters

    @property
    def seed(self) -> int | None:
        """
        Return the random seed for the EA.
        """
        return self._seed

    @property
    @abstractmethod
    def evolutionary_actor(self) -> BaseActor:
        """
        Return the current actor derived from the evolutionary process.
        """
        pass
    
    @abstractmethod
    def set_seed(self, seed: int | None) -> None:
        """
        Set the random seed for the EA and its every component that uses randomness.
        """
        self._seed = seed
        ...  # Use the seed whereever necessary.

    @abstractmethod
    def update_evolution_state_based_on_rl(self, reinforcement_actor: BaseActor) -> None:
        """
        Update the state of the EA based on the actor obtained from reinforcement learning.
        """
        pass

    @abstractmethod
    def evolve(self) -> int:
        """
        Perform a single evolution step in the EA algorithm.
        
        Returns
        -------
            int
                The number of interactions with the environment during this evolution step.
        """
        pass
